import {
  BlockPermutation,
  Container,
  Direction,
  Entity,
  EntityEquippableComponent,
  EntityInventoryComponent,
  EquipmentSlot,
  GameMode,
  ItemStack,
  system,
  world,
} from '@minecraft/server';
import { CustomEntityBucketItems } from '../item/CustomEntityBucketItems.js';
import { LocalizationParser } from '../utils/LocalizationParser.js';

export class AbstractBucketableEntity {
  static ON_BUCKET_EVENT = 'sf_nba:bucketable_entity_interaction';

  /**
   * A new instance of an entity that can be bucketable. All behaviors like vanilla buckets are applied here.
   * @param {string} identifier The enti'ty identifier.
   * @param {CustomEntityBucketItems|string} bucketVariations All the entity bucket variations.
   * @param {string} bucketableOn The bucket this entity can be bucketable. For example, for fish, 'minecraft:water_bucket' is used.
   * @param {string} liquid The block to be placed when unbucket the entity.
   * @param {{ empty: string, fill: string }} sounds Fill and Empty sounds.
   */
  constructor(identifier, bucketVariations, bucketableOn, liquid, sounds) {
    this.identifier = identifier;
    this.bucketVariations = bucketVariations;
    this.bucketableOn = bucketableOn || 'minecraft:water_bucket';
    this.liquid = liquid;
    this.sounds = sounds || { empty: '', fill: '' };

    this.onBucketEntity();
    this.onUnbucketEntity();
  }

  /**
   * Triggered when the entity is collected with a bucket.
   *
   * **Note:** Update this mechanic to use `PlayerInteractWithEntityAfterEvent` for a more precise interaction in the future.
   * This event is currently on experimental.
   */
  onBucketEntity() {
    system.afterEvents.scriptEventReceive.subscribe((e) => {
      const entity = e.sourceEntity;

      if (entity == null) return;
      if (entity.typeId === undefined) return;
      if (entity.typeId !== this.identifier) return;
      if (e.id !== AbstractBucketableEntity.ON_BUCKET_EVENT) return;

      const nearbyPlayer = entity.dimension.getPlayers({
        location: entity.location,
        closest: 4,
        maxDistance: 8,
      });

      let source = {};
      for (let player of nearbyPlayer) {
        /** @type {EntityEquippableComponent} */
        const equipments = player.getComponent(EntityEquippableComponent.componentId);
        const mainhandItem = equipments.getEquipment(EquipmentSlot.Mainhand);

        if (mainhandItem.typeId !== this.bucketableOn) return;

        source = {
          player,
          equipments,
          mainhandItem,
        };
      }

      const mainhandAmount = source.mainhandItem.amount - 1;
      world.playSound(this.sounds.fill, entity.location);

      if (mainhandAmount !== 0) {
        source.equipments.setEquipment(
          EquipmentSlot.Mainhand,
          new ItemStack(source.mainhandItem.typeId, source.mainhandItem.amount - 1)
        );
      } else {
        source.player.runCommand(`/replaceitem entity @s slot.weapon.mainhand 0 minecraft:air 1 0`);
        source.equipments.setEquipment(EquipmentSlot.Mainhand, this.createBucketItem(entity));
        return;
      }

      /** @type {Container} */
      const playerInventory = source.player.getComponent(EntityInventoryComponent.componentId).container;
      playerInventory.addItem(this.createBucketItem(entity));

      // The entity should drop the equipment when bucket! - We can't do this now.
    });
  }

  /**
   * Creates a new Bucket item for a entity.
   * @param {Entity} entity The entity to be placed inside the Bucket.
   * @returns {ItemStack} A instance of new 'Bucket of' item.
   */
  createBucketItem(entity) {
    const variant = entity?.getProperty('sf_nba:variant') || 0;
    const bucketId = !this.bucketHasVariations()
      ? this.bucketVariations
      : this.bucketVariations.getByValue(variant) || 'minecraft:bucket';
    const entityName = entity.nameTag;

    const bucket = new ItemStack(bucketId, 1);

    const bucketedEntityData = {}; // Entity nameTag and health data obj
    bucketedEntityData.health = entity.getComponent('health')?.currentValue;

    if (entityName) {
      bucketedEntityData.nameTag = entityName;
      bucket.nameTag = LocalizationParser.getParsedTranslatedText(entity, 'item.bucketCustomEntity.name', entityName);
    }

    // Save entity nameTag and health
    bucket.setDynamicProperty('sf_nba:bucketed_entity_data', JSON.stringify(bucketedEntityData));

    entity.remove();
    return bucket;
  }

  /**
   * Triggered when the entity is dropped from a bucket
   */
  onUnbucketEntity() {
    world.afterEvents.itemUseOn.subscribe((e) => {
      system.run(() => {
        const player = e.source;
        const clickedBlock = e.block;
        const clickedFace = e.blockFace;

        // Using "+ 0.5" the entity will be spawned in the center of the block.
        const spawnAt = {
          x:
            clickedFace === Direction.East
              ? clickedBlock.x + 1 + 0.5
              : clickedFace === Direction.West
                ? clickedBlock.x - 1 + 0.5
                : clickedBlock.x + 0.5,
          y:
            clickedFace === Direction.Up
              ? clickedBlock.y + 1
              : clickedFace === Direction.Down
                ? clickedBlock.y - 1
                : clickedBlock.y,
          z:
            clickedFace === Direction.North
              ? clickedBlock.z - 1 + 0.5
              : clickedFace === Direction.South
                ? clickedBlock.z + 1 + 0.5
                : clickedBlock.z + 0.5,
        };

        const item = e.itemStack;
        let entitySpawned;

        if (
          !item.typeId.startsWith('sf_nba') ||
          !item.typeId.endsWith(this.bucketVariations.parentId || this.bucketVariations)
        )
          return;

        if (!player.dimension.getBlock(spawnAt).isAir && !player.dimension.getBlock(spawnAt).isLiquid) {
          player.getComponent(EntityEquippableComponent.componentId).setEquipment(EquipmentSlot.Mainhand, item);
          return;
        }

        if (this.bucketHasVariations()) {
          const variation = this.bucketVariations.getVariation(item.typeId);
          entitySpawned = player.dimension.spawnEntity(`${this.identifier}<sf_nba:set_variant_${variation}>`, spawnAt);
        } else {
          entitySpawned = player.dimension.spawnEntity(this.identifier, spawnAt);
        }

        // Get entity data from bucket
        const bucketedEntityData = JSON.parse(item.getDynamicProperty('sf_nba:bucketed_entity_data') ?? null);
        if (bucketedEntityData?.nameTag) {
          entitySpawned.nameTag = bucketedEntityData.nameTag;
        } else if (item.nameTag) { // Keeping old code here to avoid breaking changes on buckets saved before.
          const prefix = LocalizationParser.getTranlatedTextAndRemoveOcurrencies(entitySpawned, 'item.bucketCustomEntity.name');
          entitySpawned.nameTag = item.nameTag.replace(prefix, '');
        }

        if (bucketedEntityData?.health) {
          entitySpawned.getComponent('health')?.setCurrentValue(bucketedEntityData.health);
        }

        if (this.liquid) {
          player.dimension.getBlock(spawnAt).setPermutation(BlockPermutation.resolve(this.liquid));
        }

        world.playSound(this.sounds.empty, spawnAt);

        if (player.matches({ gameMode: GameMode.creative })) return;

        player
          .getComponent(EntityEquippableComponent.componentId)
          .setEquipment(EquipmentSlot.Mainhand, new ItemStack('minecraft:bucket', 1));
      });
    });
  }

  /**
   * Checks if the entity has multiple bucket variations.
   * @returns {boolean}
   */
  bucketHasVariations() {
    return typeof this.bucketVariations === 'function';
  }
}
